//+------------------------------------------------------------------+
//|                                              ProfitTrail Pro.mq4 |
//|                             Copyright 2024, Trade Smart FX Tools |
//|                                https://tradesmartfxtools.online/ |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, Trade Smart FX Tools"
#property link      "https://tradesmartfxtools.online"
#property version   "1.00"
#property strict

// Input parameters for trailing stop
input int activationPips = 20;    // Pips after which trailing stop activates
input int trailingDistance = 15;  // Distance between stop loss and price in pips
input string Updated_version = "https://tradesmartfxtools.online/Ea-Info?ea=ProfitTrail";  // Download Multi-Level Trailing Stop

// Global variables
double pointMultiplier;          // Multiplier to calculate pips (depends on broker)
input color buy_color = clrBlue;
input color sell_color = clrRed;

// Array to store whether the alert has been triggered for each order
bool alertTriggered[];

string labelName = "tradesmartfxtools.online";
string labelText = "EA by tradesmartfxtools.online";
int labelFontSize = 18;
color labelColor = Yellow;
int spaceFromBottom = 50;

// Additional text for the updated version
string updatedLabelName = "updated_version_label";
string updatedLabelText = "Download Updated version at tradesmartfxtools.online";
int updatedLabelFontSize = 12;  // Smaller font size for the updated version text
color updatedLabelColor = White;
int updatedSpaceFromBottom = 20;  // Position it below the main label

void createOrUpdateLabels()
{
    // Create the main label
    if (ObjectFind(0, labelName) == -1)
    {
        ObjectCreate(0, labelName, OBJ_LABEL, 0, 0, 0);
        ObjectSetInteger(0, labelName, OBJPROP_CORNER, CORNER_LEFT_LOWER);
        ObjectSetInteger(0, labelName, OBJPROP_XDISTANCE, 10);
        ObjectSetInteger(0, labelName, OBJPROP_YDISTANCE, spaceFromBottom);
        ObjectSetInteger(0, labelName, OBJPROP_COLOR, labelColor);
        ObjectSetInteger(0, labelName, OBJPROP_FONTSIZE, labelFontSize);
        ObjectSetString(0, labelName, OBJPROP_TEXT, labelText);
    }

    // Create the updated version label below the main label
    if (ObjectFind(0, updatedLabelName) == -1)
    {
        ObjectCreate(0, updatedLabelName, OBJ_LABEL, 0, 0, 0);
        ObjectSetInteger(0, updatedLabelName, OBJPROP_CORNER, CORNER_LEFT_LOWER);
        ObjectSetInteger(0, updatedLabelName, OBJPROP_XDISTANCE, 10);
        ObjectSetInteger(0, updatedLabelName, OBJPROP_YDISTANCE, updatedSpaceFromBottom);
        ObjectSetInteger(0, updatedLabelName, OBJPROP_COLOR, updatedLabelColor);
        ObjectSetInteger(0, updatedLabelName, OBJPROP_FONTSIZE, updatedLabelFontSize);
        ObjectSetString(0, updatedLabelName, OBJPROP_TEXT, updatedLabelText);
    }
}




//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
    // Determine the point multiplier for pip calculations
    pointMultiplier = (Digits == 3 || Digits == 5) ? 10 : 1;

    // Log initialization message
    Print("Trailing Stop EA initialized. Activation: ", activationPips, 
          " pips, Distance: ", trailingDistance, " pips.");
          
           createOrUpdateLabels();

    return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
    switch (reason)
    {
        case REASON_REMOVE:
            Print("Trailing Stop EA removed by user.");
            break;
        case REASON_RECOMPILE:
            Print("Trailing Stop EA recompiled.");
            break;
        case REASON_CHARTCHANGE:
            Print("Trailing Stop EA removed due to chart change.");
            break;
        case REASON_CHARTCLOSE:
            Print("Trailing Stop EA removed due to chart close.");
            break;
        default:
            Print("Trailing Stop EA deinitialized. Reason code: ", reason);
    }
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
    ManageTrades();        // Call the function to manage trades
  
}

//+------------------------------------------------------------------+
//| Function to manage all trades                                    |
//+------------------------------------------------------------------+
void ManageTrades()
{
    for (int i = 0; i < OrdersTotal(); i++)
    {
        // Select the order
        if (OrderSelect(i, SELECT_BY_POS, MODE_TRADES))
        {
            // Ensure the order is for the current symbol
            if (OrderSymbol() == Symbol())
            {
                // Apply trailing stop to this order
                ApplyTrailingStop();
            }
        }
    }
}


//+------------------------------------------------------------------+
//| Function to apply trailing stop logic to a trade                 |
//+------------------------------------------------------------------+
void ApplyTrailingStop()
{
    // Determine if it's a BUY or SELL order
    if (OrderType() == OP_BUY)
    {
        double activationLevel = OrderOpenPrice() + activationPips * Point * pointMultiplier;
        double newStopLoss = Bid - trailingDistance * Point * pointMultiplier;

        // Check if the price has moved enough to activate the trailing stop
        if (Bid > activationLevel)
        {
            // Update the stop loss if the new stop loss is higher than the current one
            if (OrderStopLoss() < newStopLoss)
            {
                ModifyOrderStopLoss(OrderTicket(), newStopLoss);
            }
        }
    }
    else if (OrderType() == OP_SELL)
    {
        double activationLevel = OrderOpenPrice() - activationPips * Point * pointMultiplier;
        double newStopLoss = Ask + trailingDistance * Point * pointMultiplier;

        // Check if the price has moved enough to activate the trailing stop
        if (Ask < activationLevel)
        {
            // Update the stop loss if the new stop loss is lower than the current one
            if (OrderStopLoss() > newStopLoss || OrderStopLoss() == 0)
            {
                ModifyOrderStopLoss(OrderTicket(), newStopLoss);
            }
        }
    }
}

//+------------------------------------------------------------------+
//| Function to modify the stop loss of an order                     |
//+------------------------------------------------------------------+
void ModifyOrderStopLoss(int ticket, double newStopLoss)
{
    // Attempt to modify the order
    if (!OrderModify(ticket, OrderOpenPrice(), NormalizeDouble(newStopLoss, Digits),
                     OrderTakeProfit(), OrderExpiration(), clrNONE))
    {
        Print("Error modifying order: ", ticket, ". Error code: ", GetLastError());
    }
    else
    {
        Print("Trailing stop updated for order: ", ticket, " to ", DoubleToString(newStopLoss, Digits));
    }
}

